<?php

/**
 * This is the model class for table "Kampanie".
 *
 * The followings are the available columns in table 'Kampanie':
 * @property double $id
 * @property double $idRodzajKampanii
 * @property double $cenaDzien
 * @property double $cenaTydzien
 * @property double $cenaMiesiac
 * @property double $idBudynku
 * @property double $bonus
 * @property double $przyrost
 * @property string $jezyk
 */
class Kampanie extends CActiveRecord
{
	const TYPE_LOGO = -1;
	const PATH_LOGA = 'images/loga/loga_firmy';

	const PERIOD_DAY = 0;
	const PERIOD_WEEK = 1;
	const PERIOD_MONTH = 2;

	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return Kampanie the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'Kampanie';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		return array(
			array('idRodzajKampanii, idBudynku, bonus, przyrost', 'numerical'),
			array('cenaDzien, cenaTydzien, cenaMiesiac, bonus', 'numerical', 'min'=>0),
			array('jezyk', 'length', 'max'=>10),
			array('idRodzajKampanii, cenaDzien, cenaTydzien, cenaMiesiac, idBudynku, bonus, przyrost', 'required'),

			array('id, idRodzajKampanii, cenaDzien, cenaTydzien, cenaMiesiac, idBudynku, bonus, przyrost, jezyk', 'safe', 'on'=>'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		return array(
			'rodzaj'=>array(self::BELONGS_TO, 'RodzajKampanii', 'idRodzajKampanii'),
			'budynek'=>array(self::BELONGS_TO, 'Budynek', 'idBudynku'),
			'kampaniaFirmy'=>array(self::HAS_MANY, 'KampaniaFirmy', 'idKampanii'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id'=>Yii::t('Kampanie', 'ID'),
			'idRodzajKampanii'=>Yii::t('Kampanie', 'Rodzaj'),
			'cenaDzien'=>Yii::t('Kampanie', 'Cena za dzień [zł]'),
			'cenaTydzien'=>Yii::t('Kampanie', 'Cena za tydzień [zł]'),
			'cenaMiesiac'=>Yii::t('Kampanie', 'Cena za miesiąc [zł]'),
			'idBudynku'=>Yii::t('Kampanie', 'Budynek'),
			'bonus'=>Yii::t('Kampanie', 'Bonus (ilość dodatkowych zleceń)'),
			'przyrost'=>Yii::t('Kampanie', 'Przyrost (ilość dodatkowych zleceń w trakcie trwania)'),
			'jezyk'=>Yii::t('Kampanie', 'Język'),
		);
	}

	public function getCenaDzienPln()
	{
		return number_format($this->cenaDzien, 2, ',', ' ') . ' ,-';
	}

	public function getCenaTydzienPln()
	{
		return number_format($this->cenaTydzien, 2, ',', ' ') . ' ,-';
	}

	public function getCenaMiesiacPln()
	{
		return number_format($this->cenaMiesiac, 2, ',', ' ') . ' ,-';
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		$criteria = new CDbCriteria;
		$criteria->with = array('rodzaj', 'budynek');

		$criteria->compare(Yii::app()->db->quoteColumnName('t.id'), $this->id);
		$criteria->compare('LOWER("rodzaj"."nazwa")', strtolower($this->idRodzajKampanii), true);
		$criteria->compare('"cenaDzien"', $this->cenaDzien);
		$criteria->compare('"cenaTydzien"', $this->cenaTydzien);
		$criteria->compare('"cenaMiesiac"', $this->cenaMiesiac);
		$criteria->compare('LOWER("budynek"."nazwa")', strtolower($this->idBudynku), true);
		$criteria->compare('"bonus"', $this->bonus);
		$criteria->compare('LOWER("jezyk")', strtolower($this->jezyk), true);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
			'sort'=>array(
				'defaultOrder'=>'"t"."id" ASC',
			)
		));
	}

	public function getLogos()
	{
		$dziedziny = func_get_args();
		$result = array();
		foreach ($dziedziny as $dziedzina)
		{
			$files = CFileHelper::findFiles(self::PATH_LOGA.'/'.strtolower($dziedzina), array(
				'fileTypes' => array('jpg', 'jpeg', 'png', 'gif'),
				'level' => 0
			));
			foreach ($files as $file)
				$result[$dziedzina][basename($file)] = Yii::app()->baseUrl.'/'.$file;
		}
		return $result;
	}

	private function makeInvoice(Firma &$company, $title, $price)
	{
		static $csAgency = array(1 => 127, 2 => 128, 3 => 129);
		$faktura = new Faktury;
		$faktura->czyKoszt = 1;
		$faktura->idFirmy = $company->id;
		$faktura->idKontrahenta = $csAgency[$this->idBudynku];
		$faktura->dataWystawienia = new CDbExpression('SYSDATE');
		$faktura->opis = $title;
		$faktura->nrFaktury = 'F/' . date('Y/m/d/') . rand(0, 9) . rand(0, 9) . rand(0, 9);
		$faktura->stawkaVat = ParametryGry::getFloat('PODATEK_VAT', 0.23);
		$faktura->cenaNetto = (float)$price;
		$company->payOrder($faktura);
		$faktura->save(false);
		return $faktura->id;
	}

	public function purchaseLogo(Firma &$company, $dziedzina, $logo)
	{
		$path = self::PATH_LOGA . '/' . strtolower($dziedzina) . '/' . $logo;
		if (!file_exists($path) || is_dir($path))
			return false;

		$company->urlLogo = $path;

		$points = ParametryGry::getFloat('PUNKTY_KAMPANIA_DZIEN', 10)
			+ $this->cenaDzien * ParametryGry::getFloat('PUNKTY_KAMPANIA_WSP_CENY', 0.1);
		$company->transferPoints(Dziedzina::MARKETING, $points);

		return $this->makeInvoice($company, 'Zamówienie loga firmy.', $this->cenaDzien);
	}

	public function buy(Firma &$company, $period, $discount, $Price)
	{
		$start = time();
		$stop = $start;
		$price = 0;
		$points = 0;

		switch ($period)
		{
			case self::PERIOD_DAY:
				$stop += 60*60*24;
				$price += (float)$this->cenaDzien;
				$points = ParametryGry::getFloat('PUNKTY_KAMPANIA_DZIEN', 10);
				break;
			case self::PERIOD_WEEK:
				$stop += 60*60*24*7;
				$price += (float)$this->cenaTydzien;
				$points = ParametryGry::getFloat('PUNKTY_KAMPANIA_TYDZIEN', 50);
				break;
			case self::PERIOD_MONTH:
				$stop += 60*60*24*30;
				$price += (float)$this->cenaMiesiac;
				$points = ParametryGry::getFloat('PUNKTY_KAMPANIA_MIESIAC', 100);
				break;
			default:
				throw new Exception('Niewłaściwy okres.');
		}

		if ($discount)
		{
			$pctDiscount = $this->calculateDiscount($company->id);
			$price = max(round($price - $price * $pctDiscount), $Price);
		}

		$campaing = new KampaniaFirmy;
		$campaing->idFirmy = $company->id;
		$campaing->idKampanii = $this->id;
		$campaing->czasRozpoczecia = date('d-m-Y h:i:s', $start);
		$campaing->czasZakonczenia = date('d-m-Y h:i:s', $stop);
		$campaing->save(false);

		$priceMultiplay = ParametryGry::getFloat('PUNKTY_KAMPANIA_WSP_CENY', 0.1);
		$company->transferPoints(Dziedzina::MARKETING, $points + $price * $priceMultiplay);

		return array(
			'czasRozpoczecia' => strtotime($campaing->czasRozpoczecia),
			'czasZakonczenia' => strtotime($campaing->czasZakonczenia),
			'invoice' => $this->makeInvoice($company, 'Zamówienie kampanii reklamowej '
			. $this->rodzaj->nazwa . '.', $price)
		);
	}

	public function calculateDiscount($companyId)
	{
		$data = Yii::app()->db->createCommand('
			select (
				select count(*) from "KampaniaFirmy" "t"
				left join "Kampanie" "k" on "k"."id" = "t"."idKampanii"
				where "t"."idFirmy" = :idFirmy and  "k"."idBudynku" = :idAR
			) "current", (
				select count(*) from "KampaniaFirmy" "t"
				left join "Kampanie" "k" on "k"."id" = "t"."idKampanii"
				where "t"."idFirmy" = :idFirmy
			) "all", nvl((
				(
					select count(*) from "ZlecenieFirmy" "t"
					where "t"."idFirmy" = :idFirmy and "t"."czasZakonczenia" is not null and "t"."idOdpowiedzi" is not null
				) / nullif(nvl((
					select count(*) from "ZlecenieFirmy" "t"
					where "t"."idFirmy" = :idFirmy and "t"."czasZakonczenia" is not null
				), null),0)
			),null) "reputation"
			from dual
		')->queryRow(true, array(
			':idFirmy' => $companyId,
			':idAR' => $this->idBudynku
		));
		$current = (float)$data['current'];
		$all = (float)$data['all'];
		$loyality = $all && $current ? $current / $all : 1;
		$lT = (float)ParametryGry::getValue('AR_ZNIZKA_LOJALNOSC_PROG', 0.65);
		$loyality /= $lT; if ($loyality > 1) $loyality = 1;
		$reputation = (float)$data['reputation'];
		$rT = (float)ParametryGry::getValue('AR_ZNIZKA_REPUTACJA_PROG', 1);
		$reputation /= $rT; if ($reputation > 1) $reputation = 1;
		$lW = (float)ParametryGry::getValue('AR_ZNIZKA_LOJALNOSC_WAGA', 1);
		$rW = (float)ParametryGry::getValue('AR_ZNIZKA_REPUTACJA_WAGA', 1);
		$max = (float)ParametryGry::getValue('AR_ZNIZKA_MAX_PROC', 0.15);
		return $max * ($lW * $loyality + $rW * $reputation) / ($lW + $rW);
	}

	public function getLangName()
	{
		return ($this->jezyk)?"angielski":"polski";
	}

}
